class TableOfContent extends HTMLElement {
	constructor() {
		super();
	
		this.headingTags = ["h1", "h2", "h3", "h4", "h5", "h6"];

		this.pageSelector = ".js-dwp-page";
		this.mainSelector = ".js-dwp-main";

		this.listClass = "dwp-toc";
		this.listItemClass = "dwp-toc-item";

		this.linesOnEachPage = this.getAttribute("data-lines-on-each-page");
		this.startLevel = this.getAttribute("data-start-level");
		this.endLevel = this.getAttribute("data-end-level");
	
		this.tocList = document.createElement("ul");
		this.tocList.className = "dwp-toc"; 
	
		this.lineCount = 0;
		this.totalLinesCount = 0;
		this.tocPageNumber = 1;
	
		this.mainContentContainers = null;
		this.tocPage = null;
		this.tocPageContainer;
	
		this.attachShadow({ mode: "open" });
		this.shadowRoot.innerHTML = ``;
	
		this.initialize();
	}
  
	connectedCallback() {
	  	this.render();
	}
  
	initialize() {
	  	this.mainContentContainers = document.querySelectorAll(this.mainSelector);  
	  	this.tocPage = this.closest(this.pageSelector);

		//Create the temporary TOC container
	  	this.tocPageContainer = document.createElement("div");
		this.tocPage.after(this.tocPageContainer);
	}
  
	getTotalHeadings() {
	  let totalHeadings = 0;
  
	  this.mainContentContainers.forEach(mainContainer => {
		const pageContainer = mainContainer.closest(this.pageSelector);
  
		if (pageContainer.getAttribute("data-page-type") != "tableofcontent") {
		  const headings = mainContainer.querySelectorAll(this.headingTags.slice(this.startLevel, this.endLevel).toString());
		  totalHeadings += headings.length;
		}
	  });
  
	  return totalHeadings;
	}
  
	createListItem(heading) {
		const associatedPage = heading.closest(this.pageSelector);

		const headingText = heading.textContent;
		const headingLevel = parseInt(heading.tagName.substring(1)); // Extract the heading level

		//Create the list item
		const listItem = document.createElement("li");
		listItem.className = this.listItemClass + " " + this.listItemClass + "-lv-" + (headingLevel - this.startLevel);

		//Add link element
		const link = document.createElement("a");
		link.href = "#" + associatedPage.id;

		const linkTextLeftPart = document.createElement("span");
		linkTextLeftPart.innerHTML = headingText;

		const linkTextRightPart = document.createElement("span");
		linkTextRightPart.innerHTML = associatedPage.getAttribute("data-page-number");

		link.appendChild(linkTextLeftPart);
		link.appendChild(linkTextRightPart);

		listItem.appendChild(link);
	
	  	this.tocList.appendChild(listItem);
	}
  
	addTOCPage() {
		const tocPageTemplate = this.tocPage.cloneNode(true);
		tocPageTemplate.querySelector("table-of-content").remove();

		const tocPageMain = tocPageTemplate.querySelector(this.mainSelector);
		
		if (this.tocPageNumber != 1) {
			tocPageMain.innerHTML = "";
		}
		tocPageMain.appendChild(this.tocList);
		
		this.tocList = document.createElement("ul");
		this.tocList.className = this.listClass;
		
		this.tocPageContainer.appendChild(tocPageTemplate);
	}
	
	cleanup() {
		this.tocPage.insertAdjacentHTML("afterend", this.tocPageContainer.innerHTML);
		this.tocPageContainer.remove();
		this.tocPage.remove();
	}

	render() {
		this.mainContentContainers.forEach(mainContainer => {
			const pageContainer = mainContainer.closest(this.pageSelector);
	
			if (pageContainer.getAttribute("data-page-type") != "tableofcontent") {
			const headings = mainContainer.querySelectorAll(this.headingTags.slice(this.startLevel, this.endLevel).toString());
	
			headings.forEach(heading => {
				this.createListItem(heading);
	
				if (this.lineCount === (this.linesOnEachPage - 1) || this.totalLinesCount === (this.getTotalHeadings() - 1)) {
					this.addTOCPage();
		
					this.lineCount = 0;
					this.tocPageNumber++;
				} else {
					this.lineCount++;
				}
	
				this.totalLinesCount++;
			});
			}
		});

		this.cleanup();
	}
}
  
customElements.define("table-of-content", TableOfContent);